#!/usr/bin/env bash
set -eu
COMMAND_NAME=$1

if [ -z $COMMAND_NAME ]; then
  echo 'Please provide plugin name'
  exit 1
fi

CLASS_NAME=${COMMAND_NAME^}
LIB_VERSION=$(cat pluginlib/package.json | jq .version)

echo "Creating plugin \"$COMMAND_NAME\" with class \"$CLASS_NAME\""

# Template

PACKAGE_JSON=$(cat <<EOF
{
  "name": "@squeebot/${COMMAND_NAME}",
  "version": "1.0.0",
  "description": "",
  "main": "dist/index.js",
  "types": "./dist/index.d.ts",
  "exports": {
    "./package.json": "./package.json",
    ".": {
      "require": {
        "types": "./dist/index.d.ts",
        "default": "./dist/index.js"
      }
    }
  },
  "scripts": {
    "build": "npm run build:ts",
    "build:ts": "tsc",
    "prepare": "npm run build:ts"
  },
  "keywords": [],
  "author": "Evert Prants <evert@lunasqu.ee>",
  "license": "MIT",
  "type": "commonjs",
  "devDependencies": {
    "@types/node": "^24.10.2",
    "typescript": "^5.9.3"
  },
  "dependencies": {
    "@squeebot/pluginlib": ${LIB_VERSION}
  }
}

EOF
)

TSCONFIG=$(cat <<EOF
{
  // Visit https://aka.ms/tsconfig to read more about this file
  "compilerOptions": {
    // File Layout
    "rootDir": "./src",
    "outDir": "./dist",

    // Environment Settings
    // See also https://aka.ms/tsconfig/module
    "module": "nodenext",
    "target": "es6",
    "lib": ["esnext"],
    "types": ["node"],
    "moduleResolution": "nodenext",

    // Other Outputs
    "sourceMap": false,
    "declaration": true,
    "declarationMap": false,

    // Stricter Typechecking Options
    "noUncheckedIndexedAccess": true,
    "exactOptionalPropertyTypes": true,

    // Recommended Options
    "strict": true,
    "verbatimModuleSyntax": false,
    "isolatedModules": true,
    "noUncheckedSideEffectImports": true,
    "moduleDetection": "force",
    "skipLibCheck": true
  },
  "include": ["src/**/*.ts"]
}

EOF
)

GITIGNORE=$(cat <<EOF
/node_modules/
/dist/

EOF
)

NPMIGNORE=$(cat <<EOF
/src/
/.gitea

EOF
)

CLASS=$(cat <<EOF
import { Plugin } from '@squeebot/pluginlib/plugin';

class ${CLASS_NAME} extends Plugin {
  public name = ${CLASS_NAME}.name;
}

export default ${CLASS_NAME};

EOF
)

GITEA=$(cat <<EOF
name: Publish packages
on:
  push:
    tags:
      - '*'

jobs:
  publish-npm:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4
      - name: Setup Node.js
        uses: actions/setup-node@v6
        with:
          node-version: 'latest'
          registry-url: 'https://git.icynet.eu/api/packages/Squeebot/npm/'
          scope: '@squeebot'
      - run: npm install
        env:
          NODE_AUTH_TOKEN: \${{ secrets.PUB_ACCESS_TOKEN }}
      - run: npm run build
      - run: npm publish
        env:
          NODE_AUTH_TOKEN: \${{ secrets.PUB_ACCESS_TOKEN }}

EOF
)

# Build project
TARGET="$COMMAND_NAME/"
mkdir -p $TARGET/src
mkdir -p $TARGET/.gitea/workflows

echo "$CLASS" > $TARGET/src/index.ts
echo "$PACKAGE_JSON" > $TARGET/package.json
echo "$GITIGNORE" > $TARGET/.gitignore
echo "$NPMIGNORE" > $TARGET/.npmignore
echo "$TSCONFIG" > $TARGET/tsconfig.json
echo "$GITEA" > $TARGET/.gitea/workflows/publish.yml

cd $TARGET
npm install
git init
