#!/usr/bin/env bash
set -euo pipefail

PKGBUILD=$1
BASEURL=https://git.icynet.eu/api/packages/Squeebot/npm

if [ ! -f $PKGBUILD ]; then
  echo "Please provide a file"
  exit 1
fi

BASEDIR="$(cd "`dirname $PKGBUILD`" && pwd -P)"

# Read current vars from PKGBUILD
source $PKGBUILD

function get_package_src() {
  echo "$BASEURL/%40$_npm_user%2F$_npm_pkg/-/$1/$_npm_pkg-$1.tgz"
}

function get_version_src() {
  echo "$BASEURL/%40$_npm_user%2F$_npm_pkg"
}

echo "Determining the latest version"

# Get latest version from Gitea
LATEST_VERSION=$(curl -s $(get_version_src) | jq -r '.["dist-tags"].latest')

if [ $pkgver = $LATEST_VERSION ]; then
  echo "$pkgver is already the latest version"
  exit 0
fi

echo "Found new version $LATEST_VERSION, determining sha256sum"

# Get latest version checksum for the PKGBUILD
LATEST_CHECKSUM=$(curl -s $(get_package_src $LATEST_VERSION) | sha256sum | cut -f 1 -d " ")

echo "Patching the PKGBUILD"

# Patch the PKGBUILD
sed -i "/^sha256sums=(/{n;s/.*/\t'$LATEST_CHECKSUM'/}" $PKGBUILD
sed -i "/^pkgrel=/s/.*/pkgrel=1/" $PKGBUILD
sed -i "/^pkgver=/s/.*/pkgver=$LATEST_VERSION/" $PKGBUILD

makepkg -D $BASEDIR --printsrcinfo > $BASEDIR/.SRCINFO

echo "Pushing update to git"

git -C $BASEDIR add $BASEDIR/.SRCINFO $BASEDIR/PKGBUILD
git -C $BASEDIR commit -m "Change version to $LATEST_VERSION"
git -C $BASEDIR push

echo "All done!"
