#!/usr/bin/env bash
set -euo pipefail
PLUGIN=$1
NPM_USER=Squeebot
PACKAGER="${NPM_USER,,}"
PACKAGEBASE=https://git.icynet.eu/api/packages/$NPM_USER/npm

if [ -z $PLUGIN ]; then
  echo 'Please provide plugin name'
  exit 1
fi

DESTDIR=$PWD/pkgbuilds/nodejs-$PACKAGER-$PLUGIN

echo "Creating directory for package"
mkdir -p $DESTDIR

PLUGINSRC=https://git.icynet.eu/$NPM_USER/$PLUGIN

function get_version_src() {
  echo "$PACKAGEBASE/%40$NPM_USER%2F$PLUGIN"
}

function get_package_src() {
  echo "$PACKAGEBASE/%40$NPM_USER%2F$PLUGIN/-/$1/$PLUGIN-$1.tgz"
}

echo "Determining the latest version"

# Get latest version from Gitea
LATEST_VERSION=$(curl -s $(get_version_src) | jq -r '.["dist-tags"].latest')

echo "Found version $LATEST_VERSION, determining sha256sum"

# Get latest version checksum for the PKGBUILD
LATEST_CHECKSUM=$(curl -s $(get_package_src $LATEST_VERSION) | sha256sum | cut -f 1 -d " ")

SERVICE=$(cat <<EOF
[Unit]
Description=Squeebot v4 ${PLUGIN^} plugin
After=network.target nats-server.service

[Service]
Type=simple
ExecStartPre=mkdir -p %h/.config/squeebot/
ExecStart=/usr/bin/squeeboot @squeebot/$PLUGIN
WorkingDirectory=-%h/.config/squeebot/

[Install]
WantedBy=multi-user.target

EOF
)

SERVICE_CHECKSUM=$(echo "$SERVICE" | sha256sum | cut -f 1 -d " ")

PKGBUILD=$(cat <<EOF
# Maintainer: Evert Prants

_npm_user=$PACKAGER
_npm_pkg=$PLUGIN
pkgname=nodejs-\$_npm_user-\$_npm_pkg
pkgdesc="Squeebot v4 ${PLUGIN^} plugin"
pkgver=$LATEST_VERSION
pkgrel=1
url="$PLUGINSRC"
arch=(any)
license=(MIT)
makedepends=('npm')
optdepends=('nodejs-squeebot-squeeboot: Required to use the included service')
noextract=("\${_npm_pkg}-\${pkgver}.tgz")
sha256sums=(
	'$LATEST_CHECKSUM'
	'$SERVICE_CHECKSUM'
)
_filename=\$_npm_pkg-\$pkgver.tgz
source=(
	"$PACKAGEBASE/%40\$_npm_user%2F\$_npm_pkg/-/\$pkgver/\$_filename"
	"local://$PACKAGER-$PLUGIN.service"
)
function package() {
	npm set "@\${_npm_user}:registry" "$PACKAGEBASE/"
	npm i -g --prefix "\${pkgdir}/usr" "\${srcdir}/\$_filename"
	find "\${pkgdir}" -name package.json -print0 | xargs -r -0 sed -i '/_where/d'

	# npm gives ownership of ALL FILES to build user
	# https://bugs.archlinux.org/task/63396
	chown -R root:root "\${pkgdir}"

	install -Dm 644 "\${srcdir}/$PACKAGER-$PLUGIN.service" -t "\${pkgdir}/usr/lib/systemd/user"
}

EOF
)

GITIGNORE=$(cat <<EOF
src/
pkg/
*.pkg.tar.zst
*.tgz

EOF
)

GITEA=$(cat <<EOF
name: Publish packages
on:
  push:
    branches: [main]

jobs:
  publish:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4
      - name: Build archlinux package
        uses: https://git.icynet.eu/evert/arch-makepkg-action@v1
      - name: Publish archlinux package
        uses: https://git.icynet.eu/evert/gitea-publish-arch-packages@v1
        with:
          token: \${{ secrets.PUB_ACCESS_TOKEN }}
          repository: squeebot
          files: |-
            **.pkg.tar.zst

EOF
)

echo "Writing out"

cd $DESTDIR
git init

echo "$SERVICE" > "$PACKAGER-$PLUGIN.service"
echo "$PKGBUILD" > PKGBUILD
echo "$GITIGNORE" > .gitignore

mkdir -p .gitea/workflows
echo "$GITEA" > .gitea/workflows/publish.yml

echo "Building package for the first time"

makepkg --printsrcinfo > .SRCINFO
makepkg

echo "Adding to git"
git add .
git commit -m "Initial commit"

echo "All done!"
